package placement.ui;

import placement.anneal.*;
import placement.moves.*;


import javax.swing.JPanel;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;

/**


 * Title:        PlacementApplet
 * Description:  Animation of VLSI Module Placement
 * Copyright:    Copyright (c) 2001, 2003
 * Company:      Lafayette College
 * @author John Nestor
 * @version 1.0
 */

 /** code to plot MOVE history information - extended from UITempHistory */
public class UIMoveHistory extends JPanel {
    private PMoveHistory myMoveHistory = PMove.getHistory();
    
    
    private static final int TPZD_WIDTH=7;
    private static final int LEFT_OFFSET = 25;
    private static final int RIGHT_OFFSET = 15;
    private static final int TOP_OFFSET = 25;
    private static final int BOT_OFFSET = 20;
    private static final double LOG10_SCALE = Math.log(10.0);
    private static final int METRO_DISPLAY_HEIGHT = 30;
    private static final int METRO_MARGIN = 1;


    private static double log10(double x) { return Math.log(x) / LOG10_SCALE; }
    private int xoffset;
    private int yoffset;
    private int ymoffset; // offset for metropolis display
    private int curWidth;
    private int curHeight;
    private double logTempLeft;
    private double logTempRight;
    private double logCostTop;
    private double logCostBot;
    private double costScale;
    private double tempScale;
    private FontMetrics myFontMetrics;

    private boolean optMetropolisDisplay = false;

    public void setOptMetropolisDisplay(boolean o) { optMetropolisDisplay = o; }

    private int log10CostToY(double c) {
	return yoffset - (int)(costScale * (c - logCostBot));
    }

    private void drawXLabel(Graphics g, int k) {
	String xlabel = Integer.toString(k);
	int x = xoffset + k * TPZD_WIDTH - myFontMetrics.stringWidth(xlabel)/2;
	int y = ymoffset + myFontMetrics.getHeight() - myFontMetrics.getDescent();
	g.drawString(xlabel,x,y);
    }
    
    private void drawYLabel(Graphics g, int logC) {
	String ylabel = "1E" + Integer.toString(logC);
	int x = xoffset - myFontMetrics.stringWidth(ylabel) - 2;
	int y = log10CostToY(logC) /*+ myFontMetrics.getHeight()/2*/;
	g.drawString(ylabel,x,y);
    }
    
    private void drawLegend(Graphics g) {
	String legend = "Move Attempt History";
	g.setColor(Color.black);
	int x = xoffset + curWidth/2 - myFontMetrics.stringWidth(legend)/2;
	int y = yoffset - curHeight - 5;
	g.drawString(legend,x,y);
    }

    public void setHistoryLength(int samples) {
	Dimension myDimension = getSize();
	myDimension.width = LEFT_OFFSET + RIGHT_OFFSET + samples*TPZD_WIDTH;
	setPreferredSize(myDimension);
	
    }
	
	
/*	if (h.size()*TPDZ_WIDTH  > width) {  THIS ISN'T RIGHT YET - MOVE TO TOP!
	    myDimension.width = h.size()*TPZD_WIDTH + myInsets.left + myInsets.right;
	    setPreferredSize(myDimension);
	    }  */


  public void paintComponent(Graphics g) {
      if (myMoveHistory.size() == 0) return; // nothing to scale yet
      myFontMetrics = g.getFontMetrics();
      Insets insets = getInsets();
      curWidth = getWidth() - insets.left - insets.right - LEFT_OFFSET - RIGHT_OFFSET;
      curHeight = getHeight() - insets.top - insets.bottom - TOP_OFFSET - BOT_OFFSET;
      xoffset = insets.left + LEFT_OFFSET;
      ymoffset = yoffset = insets.top + curHeight + TOP_OFFSET;
      drawLegend(g);
      if (optMetropolisDisplay) {
	  yoffset = yoffset - METRO_DISPLAY_HEIGHT;
	  curHeight = curHeight - METRO_DISPLAY_HEIGHT;
      }
      logCostTop = Math.ceil(log10(myMoveHistory.getMaxCost()));
      logCostBot = Math.floor(log10(myMoveHistory.getMinCost()));
      costScale = curHeight / (logCostTop - logCostBot);

      // now draw the cost curve (draw axes after)
      
      int i;
      int tpzdX[] = new int[5];  // coordinate arrays used by drawPolyGon
      int tpzdY[] = new int[5];
      
      int curCost = myMoveHistory.getInitCost();
      
      for (i=0; i < myMoveHistory.size(); i++) {
	  int xLeft, xRight;
	  xLeft = xoffset + (i * TPZD_WIDTH);
	  xRight = xLeft + TPZD_WIDTH;
	  PMoveHistoryPoint hp = myMoveHistory.get(i);

	  // draw the metropolis display
	  if (optMetropolisDisplay) {
	      g.setColor(Color.gray);
	      g.drawRect(xLeft, ymoffset-METRO_DISPLAY_HEIGHT, TPZD_WIDTH-1, METRO_DISPLAY_HEIGHT); // draw the frame
	      if (hp.getProbability() > 1) System.out.println("AWK! Probability =" + hp.getProbability());
	      int probHeight = (int)((METRO_DISPLAY_HEIGHT - 2*METRO_MARGIN) * hp.getProbability());
	      if (hp.acceptedUphill()) {               // draw the probability bar
		  g.setColor(Color.orange);
		  g.fillRect(xLeft + METRO_MARGIN, ymoffset - METRO_MARGIN - probHeight,
			     TPZD_WIDTH  - 2*METRO_MARGIN, probHeight);
	      } else if (hp.accepted()) {
		  g.setColor(Color.green);
		  g.fillRect(xLeft + METRO_MARGIN, ymoffset - METRO_DISPLAY_HEIGHT - METRO_MARGIN,
			     TPZD_WIDTH  - 2*METRO_MARGIN, METRO_DISPLAY_HEIGHT - 2*METRO_MARGIN);
	      } else {
		  g.setColor(Color.red);
		  g.fillRect(xLeft + METRO_MARGIN, ymoffset - METRO_MARGIN - probHeight,
			     TPZD_WIDTH  - 2*METRO_MARGIN , probHeight);
	      }
	      g.setColor(Color.black);      	      // draw random number "R"
	      if (hp.acceptedUphill() || hp.rejected()) {
		  int yr = ymoffset - METRO_MARGIN -  (int)((METRO_DISPLAY_HEIGHT - 2*METRO_MARGIN) * hp.getR());
		  g.drawLine(xLeft + 1, yr, xRight-1, yr);
	      }
	  }
	  
	  // draw the cost bargraph trapezoid
	  tpzdX[0] = tpzdX[1] = tpzdX[4] = xLeft;
	  tpzdX[2] = tpzdX[3] = xRight;
	  tpzdY[0] = tpzdY[3] = tpzdY[4] = yoffset;
	  tpzdY[1] = log10CostToY(log10(curCost));

	  if (hp.accepted()) {
	      curCost += hp.getDeltaCost();
	      tpzdY[2] = log10CostToY(log10(curCost));
	      g.setColor(Color.gray);
	      g.drawPolygon(tpzdX, tpzdY, 4);
	      if (hp.getDeltaCost() > 0) g.setColor(Color.orange);
	      else g.setColor(Color.green);
	      g.fillPolygon(tpzdX, tpzdY, 4);
	  } else {
	      tpzdY[2] = log10CostToY(log10(curCost));
	      g.setColor(Color.gray);
	      g.drawPolygon(tpzdX, tpzdY, 4);
	      g.setColor(Color.white);
	      g.fillPolygon(tpzdX, tpzdY, 4);
	      // draw the "rejected" bar here
	      int barX1, barX2, barY;
	      barX1 = xoffset + (i * TPZD_WIDTH);
	      barX2 = xoffset + ((i+1) * TPZD_WIDTH);
	      barY = log10CostToY(log10(curCost + hp.getDeltaCost()));
	      g.setColor(Color.red);
	      if (barY < (insets.top + TOP_OFFSET)) {  // thick line indicates value off the scale
		  barY = insets.top + TOP_OFFSET ;
		  g.drawLine(barX1, barY+1, barX2, barY+1);
	      }
	      g.drawLine(barX1, barY, barX2, barY);
	  }
      }

      // draw horizontal tickmarks
      g.setColor(Color.gray);
      g.drawLine(xoffset, log10CostToY(logCostBot), xoffset + curWidth, log10CostToY(logCostBot));
      for (int j = (int)Math.ceil(logCostBot); j <= (int)Math.floor(logCostTop); j++) {
	  g.drawLine(xoffset,
		     log10CostToY(j),
		     xoffset + curWidth,
		     log10CostToY(j));
	  drawYLabel(g,j);
      }
      g.drawLine(xoffset, log10CostToY(logCostTop), xoffset + curWidth, log10CostToY(logCostTop));
      // draw vertical tickmarks
      for (int k = 0; k <= myMoveHistory.size(); k = k + 10) {
	  g.drawLine(xoffset + k * TPZD_WIDTH, yoffset, xoffset + k * TPZD_WIDTH, insets.top + TOP_OFFSET);
	drawXLabel(g,k);
      }
      
      
  }
    
}
